/*==============================================================================
Copyright (c) 2025 Libertas Technologies LLC
Author: Libertas Technologies LLC
libertastech.net
libertastech@proton.me
version: 1.0

This source code is distributed under the terms of the MIT License.

Permission is hereby granted, free of charge, to any person obtaining a copy of
this software and associated documentation files (the "Software"), to deal in
the Software without restriction, including without limitation the rights to
use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
the Software, and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
==============================================================================*/

#include <string>
#include <thread>
#include <vector>

class Stopwatch {
private:

    bool running; // true if the stopwatch is running
    bool stopStopwatch; // stops the stopwatch if true
    bool timeWasSeen; // false until showTheTime is called

    int hours; // hours in [time]
    int hunSeconds; // hundredth seconds in [time]
    int minutes; // minutes in [time]
    int seconds; // seconds in [time]

    /*
     * the following options are available for [format]:
     * h is hours
     * m is minutes
     * s is seconds
     * u is hundredths of a second
     *
     * Hours, minutes, and seconds will always be separated by a colon.
     * Hundredths of a second and seconds will be separated by a period.
     *
     * When h, m, or s are lowercase, no zero will be inserted before a
     * single digit value. Therefore, 1 hour, 32 minutes, 9 seconds displayed
     * with format hh:mm:ss will be 1:32:9. But with format H:M:S, it would
     * be displayed as 01:32:09. U is always lowercase because representing
     * hundredths of a second as one digit makes it look like tenths of a
     * second.
     *
     * h may be omitted when m and s are present.
     * m may be omitted when h is absent and s is present.
     * s must always be present.
     * u may be omitted.
     */
    std::string format; // the format with which to display [time]
    std::string time; // the time of the stopwatch

    std::vector<std::string> laps; // a list of saved laps, will cap at 100

    /*
     * @brief returns true if str contains sub_str
     * @param str: the std::string that may or may not contain sub_str
     * @param sub_str: the std::string that may or may not be in str
     * @return true if str contains sub_str, otherwise false
     */
    bool contains(std::string str, std::string sub_str) {
        if (str.find(sub_str) < str.length()) {
            return true;
        }
        else {
            return false;
        }
    }

    /*
     * @brief reformats [time] to make it human readable
     */
    void formatTime() {
        time = "";
        // if hours are required in the format, add them
        if (contains(format, "H") && hours < 10) {
            time.append("0");
        }
        if (contains(format, "H") || contains(format, "h")) {
            time.append(std::to_string(hours) + ":");
        }

        // if minutes are required in the format, add them
        if (contains(format, "M") && minutes < 10) {
            time.append("0");
        }
        if (contains(format, "M") || contains(format, "m")) {
            time.append(std::to_string(minutes) + ":");
        }

        // add seconds
        if (contains(format, "S") && seconds < 10) {
            time.append("0");
        }
        time.append(std::to_string(seconds));

        // if hundredths of a second are in the format, add them
        if (contains(format, "U")) {
            time.append(".");
            if (hunSeconds < 10) {
                time.append("0");
            }
            time.append(std::to_string(hunSeconds));
        }
    }

    /*
     * @brief increments the stopwatch
     */
    void tickTock() {
        running = true;
        while (!stopStopwatch) {
            if (hunSeconds < 99) {
                hunSeconds++;
            }
            else if (seconds < 59) {
                hunSeconds = 0;
                seconds ++;
            }
            else if (minutes < 59) {
                hunSeconds = 0;
                seconds = 0;
                minutes++;
            }
            else if (hours < 23) {
                hunSeconds = 0;
                seconds = 0;
                minutes = 0;
                hours++;
            }
            timeWasSeen = false;

            // if the stopwatch is at the end, stop
            if (hours == 23 && minutes == 59 && seconds == 59 &&
                hunSeconds == 99) {
                break;
            }
            std::this_thread::sleep_for(std::chrono::milliseconds(10));
        }
        running = false;
    }

public:

    Stopwatch(std::string newFormat = "H:M:S.U") {
        running = false;
        stopStopwatch = false;
        reset();
        setFormat(newFormat);
        timeWasSeen = false;
        time = "";
    }

    ~Stopwatch() {
        stop();
    }

    /*
     * @brief returns the requested lap
     * @param lap: the number of the lap to return
     * @return a string formatted according to setFormat(), an empty string if
     * lap is less than 0 or greater than the number of saved laps
     */
    std::string getLap(int lap) {
        if (lap < 0 || lap > laps.size()) {
            return "";
        }
        else if (laps.size() == 0) {
            return "";
        }
        return laps[lap];
    }

    /*
     * @brief whether the stopwatch is running
     * @return true if the stopwatch is running, otherwise false
     */
    bool isRunning() {
        return running;
    }

    /*
     * @brief resets the stopwatch to zero, if it's running, stop it and restart
     */
    void reset() {
        bool wasRunning = running;
        if (wasRunning) {
            stop();
        }
        hunSeconds = 0;
        seconds = 0;
        minutes = 0;
        hours = 0;
        laps.clear(); // delete all laps
        if (wasRunning) {
            start();
        }
    }

    /*
     * @brief saves the current time as a lap
     */
    void saveLap() {
        // if there are 100 laps, remove the first
        if (laps.size() == 100) {
            laps.erase(laps.begin());
        }
        formatTime();
        laps.push_back(time);
    }

    /*
     * @brief sets the valid of the format of the stopwatch time
     * @param newFormat: the new value of [format]
     * @return true if successful, otherwise false
     */
    bool setFormat(std::string newFormat) {
        bool formatValid = false; // whether newFormat is an acceptable form
        if (newFormat == "s" ||
            newFormat == "S" ||
            newFormat == "s.U" ||
            newFormat == "S.U" ||

            newFormat == "m:s" ||
            newFormat == "m:S" ||
            newFormat == "m:s.U" ||
            newFormat == "m:S.U" ||

            newFormat == "M:s" ||
            newFormat == "M:s.U" ||
            newFormat == "M:S" ||
            newFormat == "M:S.U" ||

            newFormat == "h:m:s" ||
            newFormat == "h:m:s.U" ||
            newFormat == "h:m:S" ||
            newFormat == "h:m:S.U" ||
            newFormat == "h:M:s" ||
            newFormat == "h:M:s.U" ||
            newFormat == "h:M:S" ||
            newFormat == "h:M:S.U" ||

            newFormat == "H:m:s" ||
            newFormat == "H:m:s.U" ||
            newFormat == "H:m:S" ||
            newFormat == "H:m:S.U" ||
            newFormat == "H:M:s" ||
            newFormat == "H:M:s.U" ||
            newFormat == "H:M:S" ||
            newFormat == "H:M:S.U") {

            formatValid = true;
        }

        if (formatValid) {
            format = newFormat;
        }

        return formatValid;
    }

    /*
     * @brief returns the saved time
     * @return [time]
     */
    std::string showTheTime() {
        timeWasSeen = true;
        formatTime();
        return time;
    }

    /*
     * @brief starts the stopwatch
     */
    void start() {
        if (!running) {
            std::thread stopwatchThread(&Stopwatch::tickTock, this);
            stopwatchThread.detach();
        }
    }

    /*
     * @brief stops the stopwatch thread
     */
    void stop() {
        stopStopwatch = true;
        while (running) {} // wait until it stops
        stopStopwatch = false;
    }

    /*
     * @brief should be called by the implementation to find out if [time] is
     * the same as it was since showTheTime() was last called
     * @return true if [time] changed since showTheTime() was last called
     */
    bool timeChanged() {
        return !timeWasSeen;
    }
};
